<?php
require_once 'services/MercadoPagoService.php';

class PaymentController {
    
    public function __construct() {
        // Auth check apenas para checkout, webhook é público
    }

    public function checkout($courseId) {
        if (!isLoggedIn()) {
            redirect('login');
        }

        global $pdo;
        $userId = $_SESSION['user_id'];

        // Buscar curso
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
        $stmt->execute(['id' => $courseId]);
        $course = $stmt->fetch();

        if (!$course) {
            echo "Curso não encontrado.";
            return;
        }

        // Verificar se já comprou
        $check = $pdo->prepare("SELECT * FROM enrollments WHERE user_id = :uid AND course_id = :cid");
        $check->execute(['uid' => $userId, 'cid' => $courseId]);
        if ($check->rowCount() > 0) {
            redirect('student/classroom/' . $courseId);
            return;
        }
        
        // Verificar se já tem pedido pendente
        $pending = $pdo->prepare("SELECT * FROM orders WHERE user_id = :uid AND course_id = :cid AND status = 'pending'");
        $pending->execute(['uid' => $userId, 'cid' => $courseId]);
        $pendingOrder = $pending->fetch();

        // Se o curso for gratuito, matricular automaticamente
        if ($course['price'] <= 0) {
            // Verificar se já existe matrícula (redundância, mas seguro)
            $checkEnroll = $pdo->prepare("SELECT COUNT(*) FROM enrollments WHERE user_id = ? AND course_id = ?");
            $checkEnroll->execute([$userId, $courseId]);
            
            if ($checkEnroll->fetchColumn() == 0) {
                // Criar matrícula
                $enroll = $pdo->prepare("INSERT INTO enrollments (user_id, course_id, progress, completed, created_at) VALUES (?, ?, 0, 0, NOW())");
                $enroll->execute([$userId, $courseId]);
                
                // Opcional: Criar registro de pedido "grátis"
                $order = $pdo->prepare("INSERT INTO orders (user_id, course_id, amount, status, payment_method, created_at) VALUES (?, ?, 0, 'approved', 'free', NOW())");
                $order->execute([$userId, $courseId]);
            }
            
            redirect('student/classroom/' . $courseId);
            return;
        }

        require 'views/payment/checkout.php';
    }

    public function process_mercadopago($courseId) {
        if (!isLoggedIn()) {
            redirect('login');
        }

        global $pdo;
        $userId = $_SESSION['user_id'];

        // Buscar curso
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
        $stmt->execute(['id' => $courseId]);
        $course = $stmt->fetch();

        if (!$course) {
            echo "Curso não encontrado.";
            return;
        }
        
        // Criar Pedido no Banco (Pending)
        $stmt = $pdo->prepare("INSERT INTO orders (user_id, course_id, amount, status, payment_method, created_at) VALUES (:uid, :cid, :amount, 'pending', 'mercadopago', NOW())");
        $stmt->execute([
            'uid' => $userId,
            'cid' => $courseId,
            'amount' => $course['price']
        ]);
        $orderId = $pdo->lastInsertId();

        // Buscar dados do usuário
        $userStmt = $pdo->prepare("SELECT * FROM users WHERE id = :id");
        $userStmt->execute(['id' => $userId]);
        $user = $userStmt->fetch();

        // Criar Preferência no Mercado Pago
        $mp = new MercadoPagoService();
        $preference = $mp->createPreference(
            ['title' => $course['title'], 'price' => $course['price']],
            ['name' => $user['name'], 'email' => $user['email']],
            $orderId // External Reference = ID do pedido
        );

        if ($preference && isset($preference['init_point'])) {
            // Redirecionar para o Mercado Pago
            header("Location: " . $preference['init_point']);
            exit;
        } else {
            echo "Erro ao gerar pagamento. Tente novamente.";
        }
    }

    public function process_pix_manual($courseId) {
        if (!isLoggedIn()) {
            redirect('login');
        }

        global $pdo;
        $userId = $_SESSION['user_id'];

        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['receipt'])) {
            // Buscar curso
            $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
            $stmt->execute(['id' => $courseId]);
            $course = $stmt->fetch();

            if (!$course) {
                echo "Curso não encontrado.";
                return;
            }

            // Upload do comprovante
            $file = $_FILES['receipt'];
            if ($file['error'] === 0) {
                $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = 'receipt_' . time() . '_' . $userId . '.' . $ext;
                $uploadPath = 'uploads/receipts/';
                
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0777, true);
                }
                
                if (move_uploaded_file($file['tmp_name'], $uploadPath . $filename)) {
                    // Criar Pedido
                    $stmt = $pdo->prepare("INSERT INTO orders (user_id, course_id, amount, status, payment_method, receipt_url, created_at) VALUES (:uid, :cid, :amount, 'pending', 'pix_manual', :receipt, NOW())");
                    $stmt->execute([
                        'uid' => $userId,
                        'cid' => $courseId,
                        'amount' => $course['price'],
                        'receipt' => $filename
                    ]);
                    
                    // Redirecionar para página de pendente
                    redirect('payment/pending');
                } else {
                    echo "Erro ao fazer upload do comprovante.";
                }
            } else {
                echo "Erro no arquivo enviado.";
            }
        }
    }

    public function webhook() {
        // Recebe notificação do Mercado Pago
        $json = file_get_contents('php://input');
        $data = json_decode($json, true);

        // Logs para debug (opcional)
        // file_put_contents('webhook_log.txt', print_r($data, true), FILE_APPEND);

        if (isset($data['type']) && $data['type'] == 'payment') {
            $paymentId = $data['data']['id'];
            
            $mp = new MercadoPagoService();
            $payment = $mp->getPayment($paymentId);

            if ($payment) {
                $status = $payment['status'];
                $externalReference = $payment['external_reference']; // ID do pedido
                $paymentMethod = $payment['payment_method_id'];

                $this->updateOrderStatus($externalReference, $status, $paymentId, $paymentMethod);
            }
        } else if (isset($_GET['topic']) && $_GET['topic'] == 'payment') {
             // Tratamento alternativo para Webhooks antigos
             $paymentId = $_GET['id'];
             $mp = new MercadoPagoService();
             $payment = $mp->getPayment($paymentId);
             
             if ($payment) {
                $status = $payment['status'];
                $externalReference = $payment['external_reference'];
                $paymentMethod = $payment['payment_method_id'];
                
                $this->updateOrderStatus($externalReference, $status, $paymentId, $paymentMethod);
             }
        }

        http_response_code(200);
        echo "OK";
    }

    private function updateOrderStatus($orderId, $status, $transactionId, $paymentMethod) {
        global $pdo;

        // Mapear status do MP para nosso BD
        // MP: approved, pending, in_process, rejected, refunded, cancelled
        // BD: pending, approved, rejected, refunded
        
        $dbStatus = 'pending';
        if ($status == 'approved') $dbStatus = 'approved';
        elseif ($status == 'rejected' || $status == 'cancelled') $dbStatus = 'rejected';
        elseif ($status == 'refunded') $dbStatus = 'refunded';

        // Atualizar Pedido
        $stmt = $pdo->prepare("UPDATE orders SET status = :status, transaction_id = :tid, payment_method = :pm, updated_at = NOW() WHERE id = :id");
        $stmt->execute([
            'status' => $dbStatus,
            'tid' => $transactionId,
            'pm' => $paymentMethod,
            'id' => $orderId
        ]);

        // Se aprovado, liberar curso (criar enrollment)
        if ($dbStatus == 'approved') {
            // Buscar dados do pedido para saber user e course
            $orderStmt = $pdo->prepare("SELECT * FROM orders WHERE id = :id");
            $orderStmt->execute(['id' => $orderId]);
            $order = $orderStmt->fetch();

            if ($order) {
                // Verificar se já existe matrícula para evitar duplicata
                $check = $pdo->prepare("SELECT * FROM enrollments WHERE user_id = :uid AND course_id = :cid");
                $check->execute(['uid' => $order['user_id'], 'cid' => $order['course_id']]);
                
                if ($check->rowCount() == 0) {
                    $enroll = $pdo->prepare("INSERT INTO enrollments (user_id, course_id, progress, completed) VALUES (:uid, :cid, 0, 0)");
                    $enroll->execute(['uid' => $order['user_id'], 'cid' => $order['course_id']]);
                }
            }
        }
    }

    public function success() {
        require 'views/payment/success.php';
    }

    public function failure() {
        require 'views/payment/failure.php';
    }

    public function pending() {
        require 'views/payment/pending.php';
    }
}
