<?php
class ProfessorController {
    public function __construct() {
        if (!isLoggedIn()) {
            redirect('login');
        }
        if ($_SESSION['user_role'] !== 'professor' && $_SESSION['user_role'] !== 'admin') {
            // Admin também pode acessar área de professor se quiser testar, ou redireciona
            // Mas rigorosamente seria só professor. Vou deixar restrito.
            if ($_SESSION['user_role'] !== 'professor') {
                redirect('home');
            }
        }
    }

    public function dashboard() {
        global $pdo;
        $userId = $_SESSION['user_id'];

        // Estatísticas
        $totalCourses = $pdo->query("SELECT COUNT(*) FROM courses WHERE teacher_id = $userId")->fetchColumn();
        
        // Total de vendas (alunos únicos em cursos do professor)
        $totalStudents = $pdo->query("
            SELECT COUNT(DISTINCT o.user_id) 
            FROM orders o 
            JOIN courses c ON o.course_id = c.id 
            WHERE c.teacher_id = $userId AND o.status = 'approved'
        ")->fetchColumn();

        // Total Ganhos (simples soma de orders approved)
        $totalEarnings = $pdo->query("
            SELECT SUM(o.amount) 
            FROM orders o 
            JOIN courses c ON o.course_id = c.id 
            WHERE c.teacher_id = $userId AND o.status = 'approved'
        ")->fetchColumn();

        require 'views/professor/dashboard.php';
    }

    public function courses() {
        global $pdo;
        $userId = $_SESSION['user_id'];
        
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE teacher_id = :uid ORDER BY created_at DESC");
        $stmt->execute(['uid' => $userId]);
        $courses = $stmt->fetchAll();

        require 'views/professor/courses.php';
    }

    public function create_course() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            global $pdo;
            $title = $_POST['title'];
            $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
            $description = $_POST['description'];
            $price = $_POST['price'];
            $categoryId = $_POST['category_id'];
            $teacherId = $_SESSION['user_id'];
            
            // Upload imagem
            $image = null;
            if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $imageName = uniqid() . '.' . $ext;
                move_uploaded_file($_FILES['image']['tmp_name'], 'uploads/' . $imageName);
                $image = $imageName;
            }

            $stmt = $pdo->prepare("INSERT INTO courses (title, slug, description, price, category_id, teacher_id, image, status, what_will_i_learn, target_audience, requirements) VALUES (:title, :slug, :description, :price, :cat, :teacher, :img, 'draft', :what_will_i_learn, :target_audience, :requirements)");
            $stmt->execute([
                'title' => $title,
                'slug' => $slug,
                'description' => $description,
                'price' => $price,
                'cat' => $categoryId,
                'teacher' => $teacherId,
                'img' => $image,
                'what_will_i_learn' => $_POST['what_will_i_learn'],
                'target_audience' => $_POST['target_audience'],
                'requirements' => $_POST['requirements']
            ]);

            redirect('professor/courses');
        }

        global $pdo;
        $categories = $pdo->query("SELECT * FROM categories ORDER BY name ASC")->fetchAll();
        require 'views/professor/create_course.php';
    }

    public function edit_course($id) {
        global $pdo;
        $userId = $_SESSION['user_id'];
        
        // Verificar se o curso pertence ao professor
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id AND teacher_id = :uid");
        $stmt->execute(['id' => $id, 'uid' => $userId]);
        $course = $stmt->fetch();

        if (!$course) {
            redirect('professor/courses');
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $title = $_POST['title'];
            $description = $_POST['description'];
            $price = $_POST['price'];
            $categoryId = $_POST['category_id'];
            $status = $_POST['status'];

            $sql = "UPDATE courses SET title = :title, description = :description, price = :price, duration = :duration, category_id = :cat, status = :status, what_will_i_learn = :what_will_i_learn, target_audience = :target_audience, requirements = :requirements";
            $params = [
                'title' => $title,
                'description' => $description,
                'price' => $price,
                'duration' => $_POST['duration'],
                'cat' => $categoryId,
                'status' => $status,
                'what_will_i_learn' => $_POST['what_will_i_learn'],
                'target_audience' => $_POST['target_audience'],
                'requirements' => $_POST['requirements'],
                'id' => $id
            ];

            if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $imageName = uniqid() . '.' . $ext;
                move_uploaded_file($_FILES['image']['tmp_name'], 'uploads/' . $imageName);
                $sql .= ", image = :img";
                $params['img'] = $imageName;
            }

            $sql .= " WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            redirect('professor/courses');
        }

        $categories = $pdo->query("SELECT * FROM categories ORDER BY name ASC")->fetchAll();
        require 'views/professor/edit_course.php';
    }

    public function modules($courseId) {
        global $pdo;
        $userId = $_SESSION['user_id'];

        // Verificar propriedade ou permissão de admin
        $course = null;
        
        // Tenta buscar como dono
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id AND teacher_id = :uid");
        $stmt->execute(['id' => $courseId, 'uid' => $userId]);
        $course = $stmt->fetch();

        // Se não achou e é admin, busca só pelo ID
        if (!$course && hasRole('admin')) {
            $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
            $stmt->execute(['id' => $courseId]);
            $course = $stmt->fetch();
        }

        if (!$course) redirect('professor/courses');

        // Reutilizar a lógica de módulos (ou redirecionar para uma view específica de professor)
        // Vou criar uma view específica para professor para manter isolamento, mas copiando a lógica do admin se tiver
        
        $stmt = $pdo->prepare("SELECT * FROM modules WHERE course_id = :cid ORDER BY position ASC");
        $stmt->execute(['cid' => $courseId]);
        $modules = $stmt->fetchAll();

        $totalSeconds = 0;

        foreach ($modules as &$mod) {
            $lstmt = $pdo->prepare("
                SELECT l.*, q.time_limit, q.passing_score 
                FROM lessons l 
                LEFT JOIN quizzes q ON l.id = q.lesson_id 
                WHERE l.module_id = :mid 
                ORDER BY l.position ASC
            ");
            $lstmt->execute(['mid' => $mod['id']]);
            $mod['lessons'] = $lstmt->fetchAll();

            foreach ($mod['lessons'] as $lesson) {
                if (!empty($lesson['duration'])) {
                    $parts = explode(':', $lesson['duration']);
                    if (count($parts) == 2) {
                        $totalSeconds += $parts[0] * 60 + $parts[1];
                    } elseif (count($parts) == 3) {
                        $totalSeconds += $parts[0] * 3600 + $parts[1] * 60 + $parts[2];
                    }
                }
            }
        }

        $totalDuration = '';
        if ($totalSeconds > 0) {
            $hours = floor($totalSeconds / 3600);
            $mins = floor(($totalSeconds / 60) % 60);
            $secs = $totalSeconds % 60;
            
            if ($hours > 0) {
                $totalDuration = sprintf("%dh %02dm %02ds", $hours, $mins, $secs);
            } else {
                $totalDuration = sprintf("%02dm %02ds", $mins, $secs);
            }
        } else {
            $totalDuration = '0m';
        }

        require 'views/professor/modules.php';
    }
    
    // Métodos para adicionar módulo/aula seriam chamados via POST na view de modules ou actions separadas
    public function add_module() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            global $pdo;
            $courseId = $_POST['course_id'];
            $title = $_POST['title'];
            
            // Verifica permissão
            $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
            $stmt->execute(['id' => $courseId]);
            $course = $stmt->fetch();

            if ($course && ($course['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
                $stmt = $pdo->prepare("INSERT INTO modules (course_id, title) VALUES (:cid, :title)");
                $stmt->execute(['cid' => $courseId, 'title' => $title]);
            }
            redirect('professor/modules/' . $courseId);
        }
    }

    private function getYouTubeDuration($url) {
        if (strpos($url, 'youtube.com') === false && strpos($url, 'youtu.be') === false) {
            return null;
        }

        $videoId = null;
        if (preg_match('/v=([a-zA-Z0-9_-]+)/', $url, $matches)) {
            $videoId = $matches[1];
        } elseif (preg_match('/youtu\.be\/([a-zA-Z0-9_-]+)/', $url, $matches)) {
            $videoId = $matches[1];
        }

        if (!$videoId) return null;

        // Tentar via cURL primeiro (mais robusto para produção)
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://www.youtube.com/watch?v=" . $videoId);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        // Timeout para não travar o servidor
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        $content = curl_exec($ch);
        curl_close($ch);

        // Fallback para file_get_contents se cURL falhar ou não retornar nada
        if (!$content) {
            $context = stream_context_create([
                'http' => [
                    'header' => 'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
                ]
            ]);
            $content = @file_get_contents("https://www.youtube.com/watch?v=" . $videoId, false, $context);
        }
        
        if ($content) {
            $seconds = 0;

            // Estratégia 1: JSON lengthSeconds
            if (preg_match('/"lengthSeconds":"(\d+)"/', $content, $matches)) {
                $seconds = intval($matches[1]);
            }
            // Estratégia 2: JSON approxDurationMs
            elseif (preg_match('/"approxDurationMs":"(\d+)"/', $content, $matches)) {
                $seconds = intval($matches[1]) / 1000;
            }
            // Estratégia 3: Meta tag duration (ISO 8601) - Ex: PT5M30S ou PT57S
            elseif (preg_match('/itemprop="duration" content="PT((\d+)M)?((\d+)S)?"/', $content, $matches)) {
                $min = isset($matches[2]) ? intval($matches[2]) : 0;
                $sec = isset($matches[4]) ? intval($matches[4]) : 0;
                // Se o formato for PT57S, o matches muda, vamos simplificar
                if (strpos($matches[0], 'M') === false) {
                     // Apenas segundos PT57S
                     preg_match('/PT(\d+)S/', $matches[0], $sMatch);
                     $seconds = isset($sMatch[1]) ? intval($sMatch[1]) : 0;
                } else {
                     $seconds = ($min * 60) + $sec;
                }
            }

            if ($seconds > 0) {
                if ($seconds < 3600) {
                    return gmdate("i:s", $seconds);
                } else {
                    return gmdate("H:i:s", $seconds);
                }
            }
        }
        
        return null;
    }

    public function sync_durations($courseId) {
        global $pdo;
        
        // Verifica permissão
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id AND teacher_id = :uid");
        $stmt->execute(['id' => $courseId, 'uid' => $_SESSION['user_id']]);
        
        if ($stmt->rowCount() > 0 || hasRole('admin')) {
            // Busca todas as aulas de vídeo deste curso
            $sql = "SELECT l.id, l.video_url FROM lessons l 
                    JOIN modules m ON l.module_id = m.id 
                    WHERE m.course_id = :cid AND l.type = 'video' AND l.video_url IS NOT NULL AND l.video_url != ''";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(['cid' => $courseId]);
            $lessons = $stmt->fetchAll();
            
            foreach ($lessons as $lesson) {
                $duration = $this->getYouTubeDuration($lesson['video_url']);
                if ($duration) {
                    $upd = $pdo->prepare("UPDATE lessons SET duration = :dur WHERE id = :lid");
                    $upd->execute(['dur' => $duration, 'lid' => $lesson['id']]);
                }
            }
        }
        
        redirect('professor/modules/' . $courseId);
    }

    public function add_lesson() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            global $pdo;
            $moduleId = $_POST['module_id'];
            $courseId = $_POST['course_id']; // Passado hidden para redirect
            $title = $_POST['title'];
            $type = $_POST['type'];
            $content = $_POST['content'] ?? '';
            $videoUrl = $_POST['video_url'] ?? '';

            $duration = null;
            if ($type == 'video' && !empty($videoUrl)) {
                $duration = $this->getYouTubeDuration($videoUrl);
            }

            // Quiz fields
            $timeLimit = $_POST['time_limit'] ?? 0;
            $passingScore = $_POST['passing_score'] ?? 0;

            // Verifica permissão (via módulo -> curso -> professor)
            $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
            $stmt->execute(['id' => $courseId]);
            $course = $stmt->fetch();
            
            if ($course && ($course['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
                // Calcular próxima posição
                $posStmt = $pdo->prepare("SELECT MAX(position) as max_pos FROM lessons WHERE module_id = ?");
                $posStmt->execute([$moduleId]);
                $maxPos = $posStmt->fetchColumn();
                $nextPos = ($maxPos !== false) ? $maxPos + 1 : 1;

                $stmt = $pdo->prepare("INSERT INTO lessons (module_id, title, type, content, video_url, duration, position) VALUES (:mid, :title, :type, :content, :vid, :duration, :pos)");
                $stmt->execute([
                    'mid' => $moduleId,
                    'title' => $title,
                    'type' => $type,
                    'content' => $content,
                    'vid' => $videoUrl,
                    'duration' => $duration,
                    'pos' => $nextPos
                ]);
                
                $lessonId = $pdo->lastInsertId();

                if ($type === 'quiz' || $type === 'exam') {
                    $stmt = $pdo->prepare("INSERT INTO quizzes (lesson_id, time_limit, passing_score) VALUES (:lid, :time, :score)");
                    $stmt->execute([
                        'lid' => $lessonId,
                        'time' => $timeLimit,
                        'score' => $passingScore
                    ]);
                }
            }
            redirect('professor/modules/' . $courseId);
        }
    }

    public function edit_module() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            global $pdo;
            $moduleId = $_POST['module_id'];
            $courseId = $_POST['course_id'];
            $title = $_POST['title'];

            // Verifica permissão
            $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
            $stmt->execute(['id' => $courseId]);
            $course = $stmt->fetch();

            if ($course && ($course['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
                $stmt = $pdo->prepare("UPDATE modules SET title = :title WHERE id = :mid");
                $stmt->execute(['title' => $title, 'mid' => $moduleId]);
            }
            redirect('professor/modules/' . $courseId);
        }
    }

    public function delete_module($id) {
        global $pdo;
        // Verificar permissão e buscar course_id para redirect
        $stmt = $pdo->prepare("SELECT m.course_id, c.teacher_id FROM modules m JOIN courses c ON m.course_id = c.id WHERE m.id = :id");
        $stmt->execute(['id' => $id]);
        $data = $stmt->fetch();

        if ($data && ($data['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
            // Apagar aulas primeiro
            $pdo->prepare("DELETE FROM lessons WHERE module_id = :mid")->execute(['mid' => $id]);
            // Apagar módulo
            $pdo->prepare("DELETE FROM modules WHERE id = :mid")->execute(['mid' => $id]);
            
            redirect('professor/modules/' . $data['course_id']);
        } else {
            redirect('professor/courses');
        }
    }

    public function move_lesson($id, $direction) {
        global $pdo;
        
        // Buscar info da aula e verificar permissão
        $stmt = $pdo->prepare("
            SELECT l.*, m.course_id, c.teacher_id 
            FROM lessons l 
            JOIN modules m ON l.module_id = m.id 
            JOIN courses c ON m.course_id = c.id 
            WHERE l.id = :id
        ");
        $stmt->execute(['id' => $id]);
        $lesson = $stmt->fetch();

        if (!$lesson || ($lesson['teacher_id'] != $_SESSION['user_id'] && !hasRole('admin'))) {
            redirect('professor/courses');
        }

        $moduleId = $lesson['module_id'];

        // Reorganizar posições para garantir sequência (1, 2, 3...)
        // Isso corrige casos onde position é 0 ou tem buracos
        $stmt = $pdo->prepare("SELECT id, position FROM lessons WHERE module_id = :mid ORDER BY position ASC, id ASC");
        $stmt->execute(['mid' => $moduleId]);
        $allLessons = $stmt->fetchAll();

        $pdo->beginTransaction();
        try {
            $currentIndex = -1;
            foreach ($allLessons as $index => $l) {
                $newPos = $index + 1;
                if ($l['id'] == $id) {
                    $currentIndex = $index;
                }
                // Atualiza posição se estiver errada
                if ($l['position'] != $newPos) {
                    $upd = $pdo->prepare("UPDATE lessons SET position = :pos WHERE id = :lid");
                    $upd->execute(['pos' => $newPos, 'lid' => $l['id']]);
                }
                // Atualiza array local para usar na troca
                $allLessons[$index]['position'] = $newPos;
            }

            // Realizar a troca
            if ($direction === 'up' && $currentIndex > 0) {
                $swapWith = $allLessons[$currentIndex - 1];
            } elseif ($direction === 'down' && $currentIndex < count($allLessons) - 1) {
                $swapWith = $allLessons[$currentIndex + 1];
            } else {
                $swapWith = null;
            }

            if ($swapWith) {
                $currentPos = $allLessons[$currentIndex]['position'];
                $swapPos = $swapWith['position'];

                $upd1 = $pdo->prepare("UPDATE lessons SET position = :pos WHERE id = :id");
                $upd1->execute(['pos' => $swapPos, 'id' => $id]);

                $upd2 = $pdo->prepare("UPDATE lessons SET position = :pos WHERE id = :id");
                $upd2->execute(['pos' => $currentPos, 'id' => $swapWith['id']]);
            }
            
            $pdo->commit();
        } catch (Exception $e) {
            $pdo->rollBack();
        }

        redirect('professor/modules/' . $lesson['course_id']);
    }

    public function edit_lesson() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            global $pdo;
            $lessonId = $_POST['lesson_id'];
            $courseId = $_POST['course_id'];
            $title = $_POST['title'];
            $type = $_POST['type'];
            $content = $_POST['content'] ?? '';
            $videoUrl = $_POST['video_url'] ?? '';

            // Quiz fields
            $timeLimit = $_POST['time_limit'] ?? 0;
            $passingScore = $_POST['passing_score'] ?? 0;

            // Verifica permissão
            $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id AND teacher_id = :uid");
            $stmt->execute(['id' => $courseId, 'uid' => $_SESSION['user_id']]);
            
            if ($stmt->rowCount() > 0 || hasRole('admin')) {
                $duration = null;
                if ($type == 'video' && !empty($videoUrl)) {
                    $duration = $this->getYouTubeDuration($videoUrl);
                }

                $sql = "UPDATE lessons SET title = :title, type = :type, content = :content, video_url = :vid";
                $params = [
                    'title' => $title,
                    'type' => $type,
                    'content' => $content,
                    'vid' => $videoUrl,
                    'lid' => $lessonId
                ];

                if ($duration) {
                    $sql .= ", duration = :duration";
                    $params['duration'] = $duration;
                }

                $sql .= " WHERE id = :lid";

                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);

                if ($type === 'quiz' || $type === 'exam') {
                    // Check if quiz exists
                    $qStmt = $pdo->prepare("SELECT id FROM quizzes WHERE lesson_id = ?");
                    $qStmt->execute([$lessonId]);
                    if ($qStmt->fetch()) {
                        $pdo->prepare("UPDATE quizzes SET time_limit = ?, passing_score = ? WHERE lesson_id = ?")
                            ->execute([$timeLimit, $passingScore, $lessonId]);
                    } else {
                        $pdo->prepare("INSERT INTO quizzes (lesson_id, time_limit, passing_score) VALUES (?, ?, ?)")
                            ->execute([$lessonId, $timeLimit, $passingScore]);
                    }

                    if (isset($_POST['save_and_manage'])) {
                        redirect('professor/quiz_builder/' . $lessonId);
                    }
                }
            }
            redirect('professor/modules/' . $courseId);
        }
    }

    public function delete_lesson($id) {
        global $pdo;
        $stmt = $pdo->prepare("
            SELECT l.id, m.course_id, c.teacher_id 
            FROM lessons l 
            JOIN modules m ON l.module_id = m.id 
            JOIN courses c ON m.course_id = c.id 
            WHERE l.id = :id
        ");
        $stmt->execute(['id' => $id]);
        $data = $stmt->fetch();

        if ($data && ($data['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
            $pdo->prepare("DELETE FROM lessons WHERE id = :id")->execute(['id' => $id]);
            redirect('professor/modules/' . $data['course_id']);
        } else {
            redirect('professor/courses');
        }
    }

    public function duplicate_lesson($id) {
        global $pdo;
        // Get original lesson
        $stmt = $pdo->prepare("
            SELECT l.*, m.course_id, c.teacher_id 
            FROM lessons l 
            JOIN modules m ON l.module_id = m.id 
            JOIN courses c ON m.course_id = c.id 
            WHERE l.id = :id
        ");
        $stmt->execute(['id' => $id]);
        $lesson = $stmt->fetch();

        if ($lesson && ($lesson['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
            // Insert copy
            $stmt = $pdo->prepare("INSERT INTO lessons (module_id, title, type, content, video_url, position) VALUES (:mid, :title, :type, :content, :vid, :pos)");
            $stmt->execute([
                'mid' => $lesson['module_id'],
                'title' => $lesson['title'] . ' (Cópia)',
                'type' => $lesson['type'],
                'content' => $lesson['content'],
                'vid' => $lesson['video_url'],
                'pos' => $lesson['position'] + 1
            ]);
            
            $newLessonId = $pdo->lastInsertId();

            // If quiz/exam, duplicate quiz data
            if ($lesson['type'] === 'quiz' || $lesson['type'] === 'exam') {
                $stmt = $pdo->prepare("SELECT * FROM quizzes WHERE lesson_id = ?");
                $stmt->execute([$id]);
                $quiz = $stmt->fetch();
                
                if ($quiz) {
                    // Duplicate quiz meta
                    $pdo->prepare("INSERT INTO quizzes (lesson_id, description, time_limit, passing_score) VALUES (?, ?, ?, ?)")
                        ->execute([$newLessonId, $quiz['description'], $quiz['time_limit'], $quiz['passing_score']]);
                    $newQuizId = $pdo->lastInsertId();

                    // Duplicate questions
                    $qStmt = $pdo->prepare("SELECT * FROM quiz_questions WHERE quiz_id = ? ORDER BY order_index");
                    $qStmt->execute([$quiz['id']]);
                    $questions = $qStmt->fetchAll();

                    foreach ($questions as $q) {
                        $pdo->prepare("INSERT INTO quiz_questions (quiz_id, question_text, question_type, points, order_index) VALUES (?, ?, ?, ?, ?)")
                            ->execute([$newQuizId, $q['question_text'], $q['question_type'], $q['points'], $q['order_index']]);
                        $newQuestionId = $pdo->lastInsertId();

                        // Duplicate options
                        $oStmt = $pdo->prepare("SELECT * FROM quiz_options WHERE question_id = ?");
                        $oStmt->execute([$q['id']]);
                        $options = $oStmt->fetchAll();

                        foreach ($options as $opt) {
                            $pdo->prepare("INSERT INTO quiz_options (question_id, option_text, is_correct) VALUES (?, ?, ?)")
                                ->execute([$newQuestionId, $opt['option_text'], $opt['is_correct']]);
                        }
                    }
                }
            }
            
            redirect('professor/modules/' . $lesson['course_id']);
        } else {
            redirect('professor/courses');
        }
    }

    public function quiz_builder($lessonId) {
        global $pdo;
        
        // Fetch lesson info regardless of user first
        $stmt = $pdo->prepare("
            SELECT l.*, q.id as quiz_id, q.time_limit, q.passing_score, m.course_id, c.teacher_id 
            FROM lessons l 
            JOIN modules m ON l.module_id = m.id 
            JOIN courses c ON m.course_id = c.id
            LEFT JOIN quizzes q ON l.id = q.lesson_id
            WHERE l.id = :id
        ");
        $stmt->execute(['id' => $lessonId]);
        $lesson = $stmt->fetch();

        if (!$lesson) {
            redirect('professor/courses');
        }

        // Permission check: Teacher OR Admin
        if ($lesson['teacher_id'] != $_SESSION['user_id'] && !hasRole('admin')) {
            redirect('professor/courses');
        }

        if ($lesson['type'] !== 'quiz' && $lesson['type'] !== 'exam') {
            redirect('professor/modules/' . $lesson['course_id']);
        }

        // Fetch questions
        $questions = [];
        if ($lesson['quiz_id']) {
            $stmt = $pdo->prepare("SELECT * FROM quiz_questions WHERE quiz_id = ? ORDER BY order_index");
            $stmt->execute([$lesson['quiz_id']]);
            $questions = $stmt->fetchAll();

            foreach ($questions as &$q) {
                $stmt = $pdo->prepare("SELECT * FROM quiz_options WHERE question_id = ?");
                $stmt->execute([$q['id']]);
                $q['options'] = $stmt->fetchAll();
            }
            unset($q); // Break reference to avoid overwriting in view loop
        }

        require 'views/professor/quiz_builder.php';
    }

    public function add_question() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            global $pdo;
            $quizId = $_POST['quiz_id'];
            $lessonId = $_POST['lesson_id']; // For redirect
            
            // Permission check
            $stmt = $pdo->prepare("
                SELECT c.teacher_id 
                FROM lessons l 
                JOIN modules m ON l.module_id = m.id 
                JOIN courses c ON m.course_id = c.id 
                WHERE l.id = :id
            ");
            $stmt->execute(['id' => $lessonId]);
            $data = $stmt->fetch();

            if (!$data || ($data['teacher_id'] != $_SESSION['user_id'] && !hasRole('admin'))) {
                redirect('professor/courses');
            }

            $text = $_POST['question_text'];
            $type = $_POST['question_type'];
            $points = $_POST['points'];
            
            // Insert Question
            $stmt = $pdo->prepare("INSERT INTO quiz_questions (quiz_id, question_text, question_type, points) VALUES (?, ?, ?, ?)");
            $stmt->execute([$quizId, $text, $type, $points]);
            $questionId = $pdo->lastInsertId();

            // Insert Options
            if (isset($_POST['options']) && is_array($_POST['options'])) {
                $correctIndex = $_POST['correct_option'] ?? -1;
                foreach ($_POST['options'] as $index => $optText) {
                    if (trim($optText) !== '') {
                        $isCorrect = ($index == $correctIndex) ? 1 : 0;
                        $pdo->prepare("INSERT INTO quiz_options (question_id, option_text, is_correct) VALUES (?, ?, ?)")
                            ->execute([$questionId, $optText, $isCorrect]);
                    }
                }
            }
            
            redirect('professor/quiz_builder/' . $lessonId);
        }
    }

    public function delete_question($id) {
        global $pdo;
        // Basic check via join
        $stmt = $pdo->prepare("
            SELECT l.id as lesson_id, c.teacher_id 
            FROM quiz_questions qq 
            JOIN quizzes q ON qq.quiz_id = q.id 
            JOIN lessons l ON q.lesson_id = l.id
            JOIN modules m ON l.module_id = m.id
            JOIN courses c ON m.course_id = c.id
            WHERE qq.id = :qid
        ");
        $stmt->execute(['qid' => $id]);
        $data = $stmt->fetch();

        if ($data && ($data['teacher_id'] == $_SESSION['user_id'] || hasRole('admin'))) {
            $pdo->prepare("DELETE FROM quiz_questions WHERE id = ?")->execute([$id]);
            redirect('professor/quiz_builder/' . $data['lesson_id']);
        } else {
            redirect('professor/courses');
        }
    }
    


    public function students() {
        global $pdo;
        $userId = $_SESSION['user_id'];
        
        $stmt = $pdo->prepare("
            SELECT DISTINCT u.id, u.name, u.email, c.title as course_title, o.created_at as enrolled_at 
            FROM orders o 
            JOIN courses c ON o.course_id = c.id 
            JOIN users u ON o.user_id = u.id 
            WHERE c.teacher_id = :uid AND o.status = 'approved'
            ORDER BY o.created_at DESC
        ");
        $stmt->execute(['uid' => $userId]);
        $students = $stmt->fetchAll();

        require 'views/professor/students.php';
    }
}
