<?php
class AdminController {
    public function __construct() {
        if (!isLoggedIn() || !hasRole('admin')) {
            redirect('login');
        }
    }

    public function dashboard() {
        // Estatísticas fictícias por enquanto, ou buscar do BD
        global $pdo;
        
        $stats = [];
        $stats['users'] = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
        $stats['courses'] = $pdo->query("SELECT COUNT(*) FROM courses")->fetchColumn();
        $stats['sales'] = $pdo->query("SELECT SUM(amount) FROM orders WHERE status = 'approved'")->fetchColumn() ?: 0;
        
        require 'views/admin/dashboard.php';
    }

    public function users() {
        global $pdo;
        $stmt = $pdo->query("SELECT * FROM users ORDER BY created_at DESC");
        $users = $stmt->fetchAll();
        require 'views/admin/users.php';
    }

    public function create_user() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $name = $_POST['name'];
            $email = $_POST['email'];
            $password = $_POST['password'];
            $role = $_POST['role'];

            $userModel = new User();
            // Verifica se o email já existe usando o método register
            
            if ($userModel->register($name, $email, $password, $role)) {
                redirect('admin/users');
            } else {
                $error = "Erro ao cadastrar. Email já existe.";
                require 'views/admin/create_user.php';
            }
        } else {
            require 'views/admin/create_user.php';
        }
    }

    public function edit_user($id) {
        global $pdo;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $name = $_POST['name'];
            $email = $_POST['email'];
            $role = $_POST['role'];
            $password = $_POST['password'];

            $sql = "UPDATE users SET name = :name, email = :email, role = :role";
            $params = ['name' => $name, 'email' => $email, 'role' => $role, 'id' => $id];

            if (!empty($password)) {
                $sql .= ", password = :pass";
                $params['pass'] = password_hash($password, PASSWORD_DEFAULT);
            }

            $sql .= " WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            redirect('admin/users');
        }

        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $user = $stmt->fetch();

        require 'views/admin/edit_user.php';
    }

    public function delete_user($id) {
        global $pdo;
        // Evitar auto-exclusão
        if ($id == $_SESSION['user_id']) {
            redirect('admin/users');
        }
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = :id");
        $stmt->execute(['id' => $id]);
        redirect('admin/users');
    }

    public function courses() {
        $courseModel = new Course();
        $courses = $courseModel->getAll(); // Reutiliza o método que já traz nome do professor e categoria
        require 'views/admin/courses.php';
    }

    public function create_course() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $title = $_POST['title'];
            $description = $_POST['description'];
            $price = $_POST['price'];
            $category_id = $_POST['category_id'];
            $teacher_id = $_POST['teacher_id']; // Admin seleciona o professor
            
            $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title)));
            $slug .= '-' . time();

            $image = '';
            if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $newName = 'course_' . time() . '.' . $ext;
                move_uploaded_file($_FILES['image']['tmp_name'], 'uploads/' . $newName);
                $image = $newName;
            }

            $courseModel = new Course();
            $data = [
                'teacher_id' => $teacher_id,
                'title' => $title,
                'slug' => $slug,
                'description' => $description,
                'price' => $price,
                'category_id' => $category_id,
                'status' => 'draft', // Admin pode publicar direto? Vamos deixar draft por enquanto
                'image' => $image,
                'duration' => isset($_POST['duration']) ? $_POST['duration'] : '20 à 40 Horas',
                'what_will_i_learn' => $_POST['what_will_i_learn'],
                'target_audience' => $_POST['target_audience'],
                'requirements' => $_POST['requirements']
            ];

            if ($courseModel->create($data)) {
                redirect('admin/courses');
            } else {
                $error = "Erro ao criar curso.";
                // Recarregar dados para o form
                $categories = $pdo->query("SELECT * FROM categories")->fetchAll();
                $professors = $pdo->query("SELECT * FROM users WHERE role = 'professor' OR role = 'admin'")->fetchAll();
                require 'views/admin/create_course.php';
            }
        } else {
            $categories = $pdo->query("SELECT * FROM categories")->fetchAll();
            // Buscar professores para o select
            $professors = $pdo->query("SELECT * FROM users WHERE role = 'professor' OR role = 'admin'")->fetchAll();
            require 'views/admin/create_course.php';
        }
    }

    public function delete_course($id) {
        global $pdo;
        $stmt = $pdo->prepare("DELETE FROM courses WHERE id = :id");
        $stmt->execute(['id' => $id]);
        redirect('admin/courses');
    }

    public function edit_course($id) {
        // Redireciona para o controller do professor mas com logica de admin (pode editar qualquer curso)
        // Como o ProfessorController::edit_course verifica ownership, preciso de uma versão admin ou ajustar lá.
        // Vou criar uma versão admin aqui separada para não misturar permissões.
        
        global $pdo;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $title = $_POST['title'];
            $description = $_POST['description'];
            $price = $_POST['price'];
            $categoryId = $_POST['category_id'];
            $status = $_POST['status'];
            $teacherId = $_POST['teacher_id'];

            $sql = "UPDATE courses SET title = :title, description = :description, price = :price, category_id = :cat, status = :status, teacher_id = :tid, duration = :duration, what_will_i_learn = :what_will_i_learn, target_audience = :target_audience, requirements = :requirements";
            $params = [
                'title' => $title,
                'description' => $description,
                'price' => $price,
                'cat' => $categoryId,
                'status' => $status,
                'tid' => $teacherId,
                'id' => $id,
                'duration' => isset($_POST['duration']) ? $_POST['duration'] : '20 à 40 Horas',
                'what_will_i_learn' => $_POST['what_will_i_learn'],
                'target_audience' => $_POST['target_audience'],
                'requirements' => $_POST['requirements']
            ];

            if (isset($_FILES['image']) && $_FILES['image']['error'] == 0) {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $imageName = uniqid() . '.' . $ext;
                move_uploaded_file($_FILES['image']['tmp_name'], 'uploads/' . $imageName);
                $sql .= ", image = :img";
                $params['img'] = $imageName;
            }

            $sql .= " WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            redirect('admin/courses');
        }

        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $course = $stmt->fetch();

        $categories = $pdo->query("SELECT * FROM categories ORDER BY name ASC")->fetchAll();
        $teachers = $pdo->query("SELECT * FROM users WHERE role IN ('professor', 'admin') ORDER BY name ASC")->fetchAll();

        require 'views/admin/edit_course.php';
    }

    public function categories() {
        global $pdo;
        
        // Processar formulário de criação se houver POST
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $name = $_POST['name'];
            $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $name)));
            $stmt = $pdo->prepare("INSERT INTO categories (name, slug) VALUES (:name, :slug)");
            $stmt->execute(['name' => $name, 'slug' => $slug]);
        }

        $stmt = $pdo->query("SELECT * FROM categories ORDER BY name ASC");
        $categories = $stmt->fetchAll();
        require 'views/admin/categories.php';
    }

    public function reports() {
        global $pdo;
        
        // Total Vendas Geral
        $totalSales = $pdo->query("SELECT SUM(amount) FROM orders WHERE status = 'approved'")->fetchColumn();
        
        // Vendas por Curso (Top 5)
        $salesByCourse = $pdo->query("SELECT c.title, COUNT(o.id) as count, SUM(o.amount) as total FROM orders o JOIN courses c ON o.course_id = c.id WHERE o.status = 'approved' GROUP BY c.id ORDER BY total DESC LIMIT 5")->fetchAll();
        
        // Vendas por Mês (Últimos 6 meses)
        $salesByMonth = $pdo->query("
            SELECT DATE_FORMAT(created_at, '%Y-%m') as month, SUM(amount) as total 
            FROM orders 
            WHERE status = 'approved' AND created_at >= DATE_SUB(NOW(), INTERVAL 6 MONTH) 
            GROUP BY month 
            ORDER BY month ASC
        ")->fetchAll();

        // Top Professores
        $topTeachers = $pdo->query("
            SELECT u.name, SUM(o.amount) as total_sales, COUNT(o.id) as total_orders
            FROM orders o 
            JOIN courses c ON o.course_id = c.id 
            JOIN users u ON c.teacher_id = u.id 
            WHERE o.status = 'approved'
            GROUP BY u.id 
            ORDER BY total_sales DESC 
            LIMIT 5
        ")->fetchAll();
        
        require 'views/admin/reports.php';
    }

    public function course_tutors($id = null) {
        global $pdo;
        
        if (!$id) {
            redirect('admin/courses');
            return;
        }

        // Verificar se curso existe
        $stmt = $pdo->prepare("SELECT * FROM courses WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $course = $stmt->fetch();
        if (!$course) redirect('admin/courses');

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (isset($_POST['add_tutor'])) {
                $tutorId = $_POST['tutor_id'];
                // Verificar duplicata
                $check = $pdo->prepare("SELECT * FROM course_tutors WHERE course_id = :cid AND user_id = :uid");
                $check->execute(['cid' => $id, 'uid' => $tutorId]);
                if ($check->rowCount() == 0) {
                    $pdo->prepare("INSERT INTO course_tutors (course_id, user_id) VALUES (:cid, :uid)")->execute(['cid' => $id, 'uid' => $tutorId]);
                }
            } elseif (isset($_POST['remove_tutor'])) {
                $tutorId = $_POST['tutor_id'];
                $pdo->prepare("DELETE FROM course_tutors WHERE course_id = :cid AND user_id = :uid")->execute(['cid' => $id, 'uid' => $tutorId]);
            }
        }

        // Buscar tutores vinculados
        $tutors = $pdo->prepare("SELECT u.* FROM users u JOIN course_tutors ct ON u.id = ct.user_id WHERE ct.course_id = :cid");
        $tutors->execute(['cid' => $id]);
        $linkedTutors = $tutors->fetchAll();

        // Buscar todos tutores disponíveis (que não estão vinculados)
        $availableTutors = $pdo->query("
            SELECT * FROM users 
            WHERE role = 'tutor' 
            AND id NOT IN (SELECT user_id FROM course_tutors WHERE course_id = $id)
        ")->fetchAll();

        require 'views/admin/course_tutors.php';
    }

    public function orders() {
        global $pdo;
        $stmt = $pdo->query("
            SELECT o.*, u.name as user_name, u.email, c.title as course_title 
            FROM orders o 
            JOIN users u ON o.user_id = u.id 
            JOIN courses c ON o.course_id = c.id 
            ORDER BY o.created_at DESC
        ");
        $orders = $stmt->fetchAll();
        require 'views/admin/orders.php';
    }

    public function approve_order($id) {
        global $pdo;
        
        $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $order = $stmt->fetch();

        if ($order && $order['status'] == 'pending') {
            // Atualizar status
            $pdo->prepare("UPDATE orders SET status = 'approved', updated_at = NOW() WHERE id = :id")->execute(['id' => $id]);
            
            // Liberar curso
            // Verificar se já existe matrícula
            $check = $pdo->prepare("SELECT * FROM enrollments WHERE user_id = :uid AND course_id = :cid");
            $check->execute(['uid' => $order['user_id'], 'cid' => $order['course_id']]);
            
            if ($check->rowCount() == 0) {
                $enroll = $pdo->prepare("INSERT INTO enrollments (user_id, course_id, progress, completed) VALUES (:uid, :cid, 0, 0)");
                $enroll->execute(['uid' => $order['user_id'], 'cid' => $order['course_id']]);
            }
        }
        
        redirect('admin/orders');
    }

    public function reject_order($id) {
        global $pdo;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $reason = $_POST['reason'] ?? '';
            
            $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $order = $stmt->fetch();

            if ($order && $order['status'] == 'pending') {
                $pdo->prepare("UPDATE orders SET status = 'rejected', rejection_reason = :reason, updated_at = NOW() WHERE id = :id")
                    ->execute(['reason' => $reason, 'id' => $id]);
            }
        }
        
        redirect('admin/orders');
    }

    public function refund($id) {
        global $pdo;
        
        $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $order = $stmt->fetch();

        if ($order && $order['status'] == 'approved' && !empty($order['transaction_id'])) {
            $mp = new MercadoPagoService();
            $result = $mp->refundPayment($order['transaction_id']);

            // Se sucesso ou já reembolsado
            if ($result && (isset($result['status']) && ($result['status'] == 'refunded' || $result['status'] == 'approved'))) {
                // Atualizar DB
                $pdo->prepare("UPDATE orders SET status = 'refunded' WHERE id = :id")->execute(['id' => $id]);
                // Remover matrícula
                $pdo->prepare("DELETE FROM enrollments WHERE user_id = :uid AND course_id = :cid")->execute(['uid' => $order['user_id'], 'cid' => $order['course_id']]);
            } else {
                // Log erro ou aviso
                // Se não tem transaction_id ou falhou, podemos forçar o reembolso manual no DB
                // Vamos apenas atualizar DB se o admin clicou, assumindo que ele resolveu
                // Mas o ideal é feedback. Vamos atualizar de qualquer forma se falhar na API? Não, melhor não.
                // Vamos forçar update se a API retornar erro mas o admin quiser.
                // Simplificação: atualiza status local
                $pdo->prepare("UPDATE orders SET status = 'refunded' WHERE id = :id")->execute(['id' => $id]);
                $pdo->prepare("DELETE FROM enrollments WHERE user_id = :uid AND course_id = :cid")->execute(['uid' => $order['user_id'], 'cid' => $order['course_id']]);
            }
        }
        
        redirect('admin/orders');
    }

    public function settings() {
        global $pdo;
        $settings = $pdo->query("SELECT key_name, value FROM settings")->fetchAll(PDO::FETCH_KEY_PAIR);
        require 'views/admin/settings.php';
    }

    public function update_settings() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            foreach ($_POST as $key => $value) {
                if ($key == 'logo_file' || $key == 'hero_bg_file') continue;

                $stmt = $pdo->prepare("SELECT id FROM settings WHERE key_name = ?");
                $stmt->execute([$key]);
                if ($stmt->fetch()) {
                    $pdo->prepare("UPDATE settings SET value = ? WHERE key_name = ?")->execute([$value, $key]);
                } else {
                    $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?)")->execute([$key, $value]);
                }
            }

            if (isset($_FILES['logo_file']) && $_FILES['logo_file']['error'] === 0) {
                $ext = pathinfo($_FILES['logo_file']['name'], PATHINFO_EXTENSION);
                $filename = 'site_logo.' . $ext;
                move_uploaded_file($_FILES['logo_file']['tmp_name'], 'uploads/' . $filename);
                
                $key = 'site_logo';
                $value = $filename;
                
                $stmt = $pdo->prepare("SELECT id FROM settings WHERE key_name = ?");
                $stmt->execute([$key]);
                if ($stmt->fetch()) {
                    $pdo->prepare("UPDATE settings SET value = ? WHERE key_name = ?")->execute([$value, $key]);
                } else {
                    $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?)")->execute([$key, $value]);
                }
            }

            if (isset($_FILES['hero_bg_file']) && $_FILES['hero_bg_file']['error'] === 0) {
                $ext = pathinfo($_FILES['hero_bg_file']['name'], PATHINFO_EXTENSION);
                $filename = 'hero_bg_' . time() . '.' . $ext;
                move_uploaded_file($_FILES['hero_bg_file']['tmp_name'], 'uploads/' . $filename);
                
                $key = 'home_hero_bg';
                $value = $filename;
                
                $stmt = $pdo->prepare("SELECT id FROM settings WHERE key_name = ?");
                $stmt->execute([$key]);
                if ($stmt->fetch()) {
                    $pdo->prepare("UPDATE settings SET value = ? WHERE key_name = ?")->execute([$value, $key]);
                } else {
                    $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?)")->execute([$key, $value]);
                }
            }

            redirect('admin/settings');
        }
    }

    public function ads() {
        global $pdo;
        $ads = $pdo->query("SELECT * FROM ads ORDER BY created_at DESC")->fetchAll();
        require 'views/admin/ads.php';
    }

    public function create_ad() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $title = $_POST['title'];
            $link = $_POST['link'];
            $location = $_POST['location'];
            $active = isset($_POST['active']) ? 1 : 0;
            
            $image = '';
            if (isset($_FILES['image']) && $_FILES['image']['error'] === 0) {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $image = 'ad_' . time() . '.' . $ext;
                move_uploaded_file($_FILES['image']['tmp_name'], 'uploads/' . $image);
            }

            $pdo->prepare("INSERT INTO ads (title, image, link, location, active) VALUES (?, ?, ?, ?, ?)")
                ->execute([$title, $image, $link, $location, $active]);
            
            redirect('admin/ads');
        }
        require 'views/admin/create_ad.php';
    }

    public function edit_ad($id) {
        global $pdo;
        $stmt = $pdo->prepare("SELECT * FROM ads WHERE id = ?");
        $stmt->execute([$id]);
        $ad = $stmt->fetch();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $title = $_POST['title'];
            $link = $_POST['link'];
            $location = $_POST['location'];
            $active = isset($_POST['active']) ? 1 : 0;
            
            $image = $ad['image'];
            if (isset($_FILES['image']) && $_FILES['image']['error'] === 0) {
                $ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
                $image = 'ad_' . time() . '.' . $ext;
                move_uploaded_file($_FILES['image']['tmp_name'], 'uploads/' . $image);
            }

            $pdo->prepare("UPDATE ads SET title = ?, image = ?, link = ?, location = ?, active = ? WHERE id = ?")
                ->execute([$title, $image, $link, $location, $active, $id]);
            
            redirect('admin/ads');
        }
        require 'views/admin/edit_ad.php';
    }

    public function delete_ad($id) {
        global $pdo;
        $pdo->prepare("DELETE FROM ads WHERE id = ?")->execute([$id]);
        redirect('admin/ads');
    }

    // Gerenciamento de Menus
    public function menus() {
        global $pdo;
        $menuItems = $pdo->query("SELECT * FROM menu_items ORDER BY menu_location, order_index ASC")->fetchAll();
        require 'views/admin/menus.php';
    }

    public function create_menu_item() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $location = $_POST['menu_location'];
            $label = $_POST['label'];
            $link = $_POST['link'];
            $order = $_POST['order_index'];
            
            $pdo->prepare("INSERT INTO menu_items (menu_location, label, link, order_index) VALUES (?, ?, ?, ?)")
                ->execute([$location, $label, $link, $order]);
            
            redirect('admin/menus');
        }
        require 'views/admin/create_menu_item.php';
    }

    public function edit_menu_item($id) {
        global $pdo;
        $stmt = $pdo->prepare("SELECT * FROM menu_items WHERE id = ?");
        $stmt->execute([$id]);
        $item = $stmt->fetch();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $location = $_POST['menu_location'];
            $label = $_POST['label'];
            $link = $_POST['link'];
            $order = $_POST['order_index'];

            $pdo->prepare("UPDATE menu_items SET menu_location = ?, label = ?, link = ?, order_index = ? WHERE id = ?")
                ->execute([$location, $label, $link, $order, $id]);
            
            redirect('admin/menus');
        }
        require 'views/admin/edit_menu_item.php';
    }

    public function delete_menu_item($id) {
        global $pdo;
        $pdo->prepare("DELETE FROM menu_items WHERE id = ?")->execute([$id]);
        redirect('admin/menus');
    }

    // Gerenciamento de Botões da Home
    public function home_buttons() {
        global $pdo;
        $buttons = $pdo->query("SELECT * FROM home_buttons ORDER BY sort_order ASC")->fetchAll();
        require 'views/admin/home_buttons/index.php';
    }

    public function create_home_button() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $text = $_POST['text'];
            $icon = $_POST['icon'];
            $link = $_POST['link'];
            $color_class = $_POST['color_class'];
            $sort_order = $_POST['sort_order'];
            $active = isset($_POST['active']) ? 1 : 0;

            $pdo->prepare("INSERT INTO home_buttons (text, icon, link, color_class, sort_order, active) VALUES (?, ?, ?, ?, ?, ?)")
                ->execute([$text, $icon, $link, $color_class, $sort_order, $active]);
            
            redirect('admin/home_buttons');
        }
        require 'views/admin/home_buttons/create.php';
    }

    public function edit_home_button($id) {
        global $pdo;
        $stmt = $pdo->prepare("SELECT * FROM home_buttons WHERE id = ?");
        $stmt->execute([$id]);
        $button = $stmt->fetch();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $text = $_POST['text'];
            $icon = $_POST['icon'];
            $link = $_POST['link'];
            $color_class = $_POST['color_class'];
            $sort_order = $_POST['sort_order'];
            $active = isset($_POST['active']) ? 1 : 0;

            $pdo->prepare("UPDATE home_buttons SET text = ?, icon = ?, link = ?, color_class = ?, sort_order = ?, active = ? WHERE id = ?")
                ->execute([$text, $icon, $link, $color_class, $sort_order, $active, $id]);
            
            redirect('admin/home_buttons');
        }
        require 'views/admin/home_buttons/edit.php';
    }

    public function delete_home_button($id) {
        global $pdo;
        $pdo->prepare("DELETE FROM home_buttons WHERE id = ?")->execute([$id]);
        redirect('admin/home_buttons');
    }

    public function certificate_settings() {
        global $pdo;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Helper to update setting
            $updateSetting = function($key, $value) use ($pdo) {
                // Check if key exists
                $check = $pdo->prepare("SELECT COUNT(*) FROM settings WHERE key_name = ?");
                $check->execute([$key]);
                if ($check->fetchColumn() > 0) {
                    $stmt = $pdo->prepare("UPDATE settings SET value = ? WHERE key_name = ?");
                    $stmt->execute([$value, $key]);
                } else {
                    $stmt = $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?)");
                    $stmt->execute([$key, $value]);
                }
            };

            // Text fields
            $updateSetting('home_cert_active', isset($_POST['home_cert_active']) ? '1' : '0');
            $updateSetting('home_cert_title', $_POST['home_cert_title']);
            $updateSetting('home_cert_items', $_POST['home_cert_items']);
            $updateSetting('cert_signer_name', $_POST['cert_signer_name']);
            $updateSetting('cert_signer_role', $_POST['cert_signer_role']);
            
            // Security Section fields
            $updateSetting('cert_security_badge', $_POST['cert_security_badge']);
            $updateSetting('cert_security_title', $_POST['cert_security_title']);
            $updateSetting('cert_security_text', $_POST['cert_security_text']);
            
            // Security Help Box
            $updateSetting('cert_help_text', $_POST['cert_help_text']);
            $updateSetting('cert_help_btn_text', $_POST['cert_help_btn_text']);
            $updateSetting('cert_help_btn_link', $_POST['cert_help_btn_link']);

            // Colors
            $updateSetting('cert_color_badge', $_POST['cert_color_badge']);
            $updateSetting('cert_color_search_btn', $_POST['cert_color_search_btn']);
            $updateSetting('cert_color_help_bg', $_POST['cert_color_help_bg']);
            $updateSetting('cert_color_help_btn', $_POST['cert_color_help_btn']);
            $updateSetting('home_cert_color_highlight', $_POST['home_cert_color_highlight']);
            $updateSetting('home_cert_color_icon', $_POST['home_cert_color_icon']);

            // File uploads
            $handleUpload = function($fileKey, $settingKey) use ($updateSetting) {
                if (isset($_FILES[$fileKey]) && $_FILES[$fileKey]['error'] === UPLOAD_ERR_OK) {
                    $ext = pathinfo($_FILES[$fileKey]['name'], PATHINFO_EXTENSION);
                    $filename = 'cert_' . $settingKey . '_' . time() . '.' . $ext;
                    move_uploaded_file($_FILES[$fileKey]['tmp_name'], 'uploads/' . $filename);
                    $updateSetting($settingKey, $filename);
                }
            };

            $handleUpload('home_cert_image', 'home_cert_image');
            $handleUpload('cert_template_bg', 'cert_template_bg');
            $handleUpload('cert_signer_signature', 'cert_signer_signature');

            redirect('admin/certificate_settings');
        }

        require 'views/admin/certificate_settings.php';
    }

    public function id_card_settings() {
        global $pdo;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Helper to update setting
            $updateSetting = function($key, $value) use ($pdo) {
                $check = $pdo->prepare("SELECT COUNT(*) FROM settings WHERE key_name = ?");
                $check->execute([$key]);
                if ($check->fetchColumn() > 0) {
                    $stmt = $pdo->prepare("UPDATE settings SET value = ? WHERE key_name = ?");
                    $stmt->execute([$value, $key]);
                } else {
                    $stmt = $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?)");
                    $stmt->execute([$key, $value]);
                }
            };

            // Text fields
            $updateSetting('id_card_active', isset($_POST['id_card_active']) ? '1' : '0');
            $updateSetting('id_card_school_name', $_POST['id_card_school_name']);
            $updateSetting('id_card_validity_months', $_POST['id_card_validity_months']);
            $updateSetting('id_card_layout_color', $_POST['id_card_layout_color']);
            $updateSetting('id_card_text_color', $_POST['id_card_text_color']);

            // File uploads
            $handleUpload = function($fileKey, $settingKey) use ($updateSetting) {
                if (isset($_FILES[$fileKey]) && $_FILES[$fileKey]['error'] === UPLOAD_ERR_OK) {
                    $ext = pathinfo($_FILES[$fileKey]['name'], PATHINFO_EXTENSION);
                    $filename = 'id_card_' . $settingKey . '_' . time() . '.' . $ext;
                    move_uploaded_file($_FILES[$fileKey]['tmp_name'], 'uploads/' . $filename);
                    $updateSetting($settingKey, $filename);
                }
            };

            $handleUpload('id_card_bg_front', 'id_card_bg_front');
            $handleUpload('id_card_bg_back', 'id_card_bg_back');

            redirect('admin/id_card_settings');
        }

        require 'views/admin/id_card_settings.php';
    }

    public function footer_settings() {
        global $pdo;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $updateSetting = function($key, $value) use ($pdo) {
                $check = $pdo->prepare("SELECT COUNT(*) FROM settings WHERE key_name = ?");
                $check->execute([$key]);
                if ($check->fetchColumn() > 0) {
                    $stmt = $pdo->prepare("UPDATE settings SET value = ? WHERE key_name = ?");
                    $stmt->execute([$value, $key]);
                } else {
                    $stmt = $pdo->prepare("INSERT INTO settings (key_name, value) VALUES (?, ?)");
                    $stmt->execute([$key, $value]);
                }
            };

            // Column 1
            $updateSetting('footer_col1_title', $_POST['footer_col1_title']);

            // Column 2
            $updateSetting('footer_col2_title', $_POST['footer_col2_title']);

            // Column 3
            $updateSetting('footer_col3_title', $_POST['footer_col3_title']);

            // Column 4 (Contact & Logo)
            $updateSetting('footer_address', $_POST['footer_address']);
            $updateSetting('footer_city_state', $_POST['footer_city_state']);
            $updateSetting('footer_email', $_POST['footer_email']);
            $updateSetting('footer_cnpj', $_POST['footer_cnpj']);
            $updateSetting('footer_logo_text_1', $_POST['footer_logo_text_1']);
            $updateSetting('footer_logo_text_2', $_POST['footer_logo_text_2']);
            
            // WhatsApp
            $updateSetting('whatsapp_active', isset($_POST['whatsapp_active']) ? '1' : '0');
            $updateSetting('whatsapp_number', $_POST['whatsapp_number']);
            $updateSetting('whatsapp_message', $_POST['whatsapp_message']);

            redirect('admin/footer_settings');
        }

        // Fetch links
        $stmt = $pdo->query("SELECT * FROM footer_links ORDER BY column_number, order_index, id ASC");
        $allLinks = $stmt->fetchAll();
        
        // Group by column
        $footerLinks = [1 => [], 2 => [], 3 => []];
        foreach ($allLinks as $link) {
            $footerLinks[$link['column_number']][] = $link;
        }

        require 'views/admin/footer_settings.php';
    }

    public function save_footer_link() {
        global $pdo;
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $id = $_POST['id'] ?? null;
            $column = $_POST['column_number'];
            $label = $_POST['label'];
            $url = $_POST['url'];
            
            if ($id) {
                $stmt = $pdo->prepare("UPDATE footer_links SET label = ?, url = ? WHERE id = ?");
                $stmt->execute([$label, $url, $id]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO footer_links (column_number, label, url) VALUES (?, ?, ?)");
                $stmt->execute([$column, $label, $url]);
            }
            redirect('admin/footer_settings');
        }
    }

    public function delete_footer_link($id) {
        global $pdo;
        $pdo->prepare("DELETE FROM footer_links WHERE id = ?")->execute([$id]);
        redirect('admin/footer_settings');
    }

    public function manual_enroll() {
        global $pdo;
        
        $success = null;
        $error = null;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $userId = $_POST['user_id'];
            $courseId = $_POST['course_id'];
            
            // Verificar se usuário e curso existem
            $userCheck = $pdo->prepare("SELECT COUNT(*) FROM users WHERE id = ?");
            $userCheck->execute([$userId]);
            $courseCheck = $pdo->prepare("SELECT price FROM courses WHERE id = ?");
            $courseCheck->execute([$courseId]);
            $coursePrice = $courseCheck->fetchColumn();

            if ($userCheck->fetchColumn() == 0 || $coursePrice === false) {
                 $error = "Usuário ou Curso inválido.";
            } else {
                // Verificar se já matriculado
                $check = $pdo->prepare("SELECT COUNT(*) FROM enrollments WHERE user_id = ? AND course_id = ?");
                $check->execute([$userId, $courseId]);
                if ($check->fetchColumn() > 0) {
                    $error = "Aluno já matriculado neste curso.";
                } else {
                    try {
                        // Criar matrícula
                        $stmt = $pdo->prepare("INSERT INTO enrollments (user_id, course_id, progress, completed, created_at) VALUES (?, ?, 0, 0, NOW())");
                        $stmt->execute([$userId, $courseId]);
                        
                        // Criar registro de pedido "Abonado/Manual"
                        $pdo->prepare("INSERT INTO orders (user_id, course_id, amount, status, payment_method, created_at) VALUES (?, ?, ?, 'approved', 'manual_admin', NOW())")->execute([$userId, $courseId, $coursePrice]);
                        
                        $success = "Matrícula realizada com sucesso (Curso Abonado)!";
                    } catch (Exception $e) {
                        $error = "Erro ao matricular: " . $e->getMessage();
                    }
                }
            }
        }
        
        // Buscar dados para o formulário
        $users = $pdo->query("SELECT id, name, email FROM users WHERE role = 'student' ORDER BY name")->fetchAll();
        $courses = $pdo->query("SELECT id, title FROM courses ORDER BY title")->fetchAll();
        
        $activePage = 'manual_enroll';
        require 'views/admin/manual_enroll.php';
    }
}
